// edo.h

// Ecuaciones Diferenciales Ordinarias


// NECESITA :

//#include <stdio.h>
//#include <math.h>
//#include <stdlib.h>
  
//#include "varios.h"  
//#include "opelvect.h"
//#include "algelin.h"


double fxy_1(double x, double y);
void FxY_2(double x, double *y, double *Dy);
double Dmy_3(double x, double *Diy);
double D2y_5(double x, double y, double Dy);
void EDL2_6(double x, double *pqrs);

int Euler(double (*f)(double x, double y), double a, double b, 
        double ya, int n, double *Y);
int Heun(double (*f)(double x, double y), double a, double b, 
	 double ya, int n, double *Y);
int puntoMed(double (*f)(double x, double y), double a, double b, 
        double ya, int n, double *Y);
int RK4(double (*f)(double x, double y), double a, double b, 
        double ya, int n, double *Y);
int AdamsB(double (*f)(double x, double y), double a, double b, 
           double ya, int m, int n, double *Y);
int AdamsM(double (*f)(double x, double y), double a, double b, 
           double ya, int m, int n, double *Y, double eps);
int RK4Sist(void (*f)(double x, double *y, double *Dy), int m, 
	    double a, double b, double *ya, int n, double *Y);
int RK4m(double (*f)(double x, double *Diy), int m, 
         double a, double b, double *Diya, int n, double *Y);
int RK4_2(double (*f)(double x, double y, double Dy), double a, 
          double b, double ya, double Dya, int n, double *Y);
int disparo(double (*f)(double x, double y, double Dy), double a, 
	    double b, double ya, double yb, int n, double *Y, double eps);
int DF_CF(void (*f)(double x, double *pqrs), double a, 
          double b, double ya, double yb, int n, double *Y);

void K1234m(double (*f)(double x, double *Diy), double h, double xiah, 
            double *yi, double beta, double *K_1, double *K, int m);
void K1234_2(double (*f)(double x, double y, double Dy), double h, 
             double xiah, double *yi, double beta, double *K_1, double *K);


//==========================================================
//----------------------------------------------------------
double fxy_1(double x, double y)
{
  // funcion de ecuacion diferencial

  double Dy; // derivada de y = y'

  // y' = 2 x^2 - 4 x + y
  Dy = 2.0*x*x - 4.0*x + y;
  return Dy;
}
//----------------------------------------------------------
void FxY_2(double x, double *y, double *Dy)
{
  // funcion de un sistema de ecuaciones diferenciales

  // y1' = 2 y1 / x  + x^3 y2
  // y2' = -3 y2 / x

  // y1 = y[0]
  // y2 = y[1]

  // y1' = Dy[0]
  // y2' = Dy[1]
 
  Dy[0] =  2.0*y[0]/x + x*x*x * y[1];
  Dy[1] = -3.0*y[1]/x;
}
//----------------------------------------------------------
double Dmy_3(double x, double *Diy)
{
  // ecuacion diferencial de orden  m
  //
  // D^m y = y^(m) = f(x, y, y', ..., y^(m-1) )
  //
  // el vector  Diy  :  y,  y', y'', ..., y^(m-1)
  //  
  // En este caso, ecuacion diferencial de orden  2

  // y'' = ( 4 y - x y' )/( x*x )

  return ( 4*Diy[0] - x *Diy[1] )/ ( x*x ); 
}
//----------------------------------------------------------
double D2y_5(double x, double y, double Dy)
{
  // ecuacion diferencial de orden  2
  //
  // y'' = f(x, y, y')
  //
  // y'' = ( 4 y - x y' )/( x*x )

  return ( 4*y - x*Dy )/ ( x*x ); 
}
//----------------------------------------------------------
void EDL2_6(double x, double *pqrs)
{
  // ecuacion diferencial lineal de segundo orden
  //
  // p(x) y'' + q(x) y' + r(x) y = s(x)
  //
  // Los valores p(x), q(x), r(x), s(x)  quedan
  // en ese orde en el arreglo pqrs
  // 
  // x^2 y'' + y' + 4 x^2 y = 2 cos( 2 x)

  pqrs[0] = x*x;
  pqrs[1] = 1.0;
  pqrs[2] = 4*x*x;
  pqrs[3] = 2.0*cos(2.0*x);
}
//----------------------------------------------------------
void K1234m(double (*f)(double x, double *Diy), double h, double xiah, 
            double *yi, double beta, double *K_1, double *K, int m)
{
  // construccion  de  Ki para metodo RK4m
  //                   o sea para RK4 aplicada a  una ecuacion
  //                   diferencial de orden  m
  //
  // K = h * F( xiah, yi + beta * K_1 )
  //    
  // F es la funcion del sistema de  m  ecuaciones diferenciales
  // obtenida a partir de una ecuacion diferencial de orden  m
  // definida segun  f
  //
  
  int j;
  double *t;

  t = new double[m];

  if( beta == 0.0 ) XIglY(t, yi, m);
  else ZIglXMasAlfaY(t, yi, beta, K_1, m);
    
  for( j = 0; j <= m-2; j++ ) K[j] = h*t[j+1];
  K[m-1] = h*f(xiah, t);

  delete t;
}
//----------------------------------------------------------
void K1234_2(double (*f)(double x, double y, double Dy), double h, 
             double xiah, double *yi, double beta, double *K_1, double *K)
{
  // construccion  de  Ki para metodo RK4_2
  //                   o sea para RK4 aplicado a  una ecuacion
  //                   diferencial de orden  2
  //
  // K = h * F( xiah, yi + beta * K_1 )
  //    
  // F es la funcion del sistema de  m  ecuaciones diferenciales
  // obtenida a partir de una ecuacion diferencial de orden  m
  // definida segun  f
  //
  
  double t[2];

  if( beta == 0.0 ) XIglY(t, yi, 2);
  else ZIglXMasAlfaY(t, yi, beta, K_1, 2);

  
  K[0] = h*t[1];
  K[1] = h*f(xiah, t[0], t[1]);

}
//----------------------------------------------------------
int RK4(double (*f)(double x, double y), double a, double b, 
        double ya, int n, double *Y)
{
  // metodo de Runge-Kutta de orden 4 para la ecuacion diferencial
  // 
  //   y' = f(x,y)  en el intervalo  [a, b]
  //   y(a) = ya
  //
  // con  n >= 1  subintervalos.

  // Devuelve   1  si los datos estan bien,
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  //  Y[i] sera la aproximacion de  y( a + i h ), i = 0,1,2,...,n
  //
  //  donde
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  n+1  valores.

  double h, K1, K2, K3, K4, xi, yi, h2;
  int i;

  if( a > b ) {
    printf("ERROR: RK4: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: RK4: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: RK4: n <= 0.\n");
    return -1; 
  }

  h = (b-a)/double(n);
  h2 = h/2.0;
  Y[0] = ya;
  for( i = 0; i < n; i++) {
    xi = a + double(i)*h;
    yi = Y[i];
    K1 = h*f( xi, yi);
    K2 = h*f( xi + h2, yi+K1/2.0);
    K3 = h*f( xi + h2, yi+K2/2.0);
    K4 = h*f( xi + h,  yi+K3);
    Y[i+1] = yi + (K1 + 2.0*K2 + 2.0*K3 + K4)/6.0;

    //printf("K1234 %12.6lf%12.6lf%12.6lf%12.6lf\n",K1,K2,K3,K4);

  }
  return 1;
}
//----------------------------------------------------------
int Euler(double (*f)(double x, double y), double a, double b, 
        double ya, int n, double *Y)
{
  // metodo de Euler para la ecuacion diferencial
  // 
  //   y' = f(x,y)  en el intervalo  [a, b]
  //   y(a) = ya
  //
  // con  n >= 1  subintervalos.

  // Devuelve   1  si los datos est� bien,
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  //  Y[i] sera la aproximacion de  y( a + i h ), i = 0,1,2,...,n
  //
  //  donde
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  n+1  valores.

  double h, K1, xi, yi;
  int i;

  if( a > b ) {
    printf("ERROR: Euler: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: Euler: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: Euler: n <= 0.\n");
    return -1; 
  }

  h = (b-a)/double(n);
  Y[0] = ya;
  for( i = 0; i < n; i++) {
    xi = a + double(i)*h;
    yi = Y[i];
    K1 = h*f( xi, yi);
    Y[i+1] = yi + K1;

    //printf("K1 %16.8lf\n",K1);

  }
  return 1;
}
//----------------------------------------------------------
int puntoMed(double (*f)(double x, double y), double a, double b, 
        double ya, int n, double *Y)
{
  // metodo del punto medio para la ecuacion diferencial
  // 
  //   y' = f(x,y)  en el intervalo  [a, b]
  //   y(a) = ya
  //
  // con  n >= 1  subintervalos.

  // Devuelve   1  si los datos estan bien,
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  //  Y[i] sera la aproximacion de  y( a + i h ), i = 0,1,2,...,n
  //
  //  donde
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  n+1  valores.

  double h, K1, K2, xi, yi;
  int i;

  if( a > b ) {
    printf("ERROR: puntoMed: a > b.\n");
    return -1; 
  }

  if( n <= 0 ) {
    printf("ERROR: puntoMed: n <= 0.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: puntoMed: a = b.\n");
  }

  h = (b-a)/double(n);
  Y[0] = ya;
  for( i = 0; i < n; i++) {
    xi = a + double(i)*h;
    yi = Y[i];
    K1 = h*f( xi, yi);
    K2 = h*f( xi + h/2.0, yi+K1/2.0);
    Y[i+1] = yi + K2;


    //printf("K1 K2 %16.8lf %16.8lf\n", K1, K2);

  }
  return 1;
}
//----------------------------------------------------------
int Heun(double (*f)(double x, double y), double a, double b, 
        double ya, int n, double *Y)
{
  // metodo de Heun para la ecuacion diferencial
  // 
  //   y' = f(x,y)  en el intervalo  [a, b]
  //   y(a) = ya
  //
  // con  n >= 1  subintervalos.

  // Devuelve   1  si los datos estan bien,
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  //  Y[i] sera la aproximacion de  y( a + i h ), i = 0,1,2,...,n
  //
  //  donde
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  n+1  valores.

  double h, K1, K2, xi, yi;
  int i;

  if( a > b ) {
    printf("ERROR: Heun: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: Heun: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: Heun: n <= 0.\n");
    return -1; 
  }

  h = (b-a)/double(n);
  Y[0] = ya;
  for( i = 0; i < n; i++) {
    xi = a + double(i)*h;
    yi = Y[i];
    K1 = h*f( xi, yi);
    K2 = h*f( xi + h, yi + K1);
    Y[i+1] = yi + ( K1 + K2 )/2.0;


    //printf("K1 K2 %16.8lf %16.8lf\n", K1, K2);

  }
  return 1;
}
//----------------------------------------------------------
int RKF(double (*f)(double x, double y), double a, double b, 
        double ya, double h0, double hmin, double eps,
        double *X, double *Y, int nmax, int &n)
{
  // metodo de Runge-Kutta-Fehlberg con control de paso 
  // para la ecuacion diferencial
  // 
  //   y' = f(x,y)  en el intervalo  [a, b]
  //   y(a) = ya
  //
  // con  n >= 1  subintervalos.

  // Devuelve   1  si todo el proceso se desarrolla bien.
  //           -1  si hay errores en los datos.
  //            0  si el  h calculado es menor que  hmin
  //            2  si los arreglos X, Y no tienen espacio suficiente
  //               para almacenar los resultados
  //
  // Los valores calculados estaran en los arreglo  X, Y
  // de la siguiente forma:
  // 
  //  Y[i] sera la aproximacion de  y( X[i] ), i = 0,1,2,...,n
  //
  //
  // Los  arreglos o apuntadores  X, Y  deben estar previstos
  // para almacenar, cada uno, por lo menos  nmax  valores.

  // h0    es el paso inicial.
  // hmin  es el h minimo.
  // eps   es la tolerancia para los errores calculados,
  //       es decir, en cada iteracion, el error calculado 
  //       debe ser menor que  eps.
  // nmax  es el tama~no de los arreglos o apuntadores  X, Y.

  // n     indicara  el numero de puntos obtenidos.
  

  double h, K1, K2, K3, K4, K5, K6, x, y, y5, y6, e, C, hv;
  int res, hbien;
  double a2 = 0.25, a3 = 3.0/8, a4 = 12.0/13, a5 = 1.0, a6 = 0.5;
  double b21 = 0.25, b31 = 3.0/32, b32 = 9.0/32;
  double b41 = 1932.0/2197, b42 = -7200.0/2197, b43 = 7296.0/2197;
  double b51 = 439.0/216, b52 = -8.0, b53 = 3680.0/513, b54 = -845.0/4104;
  double b61 = -8.0/27, b62 = 2.0, b63 = -3544.0/2565, b64 = 1859.0/4104, 
    b65 = -11.0/40;
  double R51 = 25.0/216, R53 = 1408.0/2565, R54 = 2197.0/4104, R55 = -0.2;
  double R61 = 16.0/135, R63 = 6656.0/12825, R64 = 28561.0/56430, 
    R65 = -9.0/50, R66 = 2.0/55;  
  
 
  //***********************************************
  double eps0 = 1.0e-16;
  //int IRP = 0;  //indicador de resultados parciales
  //***********************************************

  //printf("a2 b2 %lf %lf\n", a2, b21);
  //printf("a3 b3 %lf %lf\n", a3, b31+b32);
  //printf("a4 b4 %lf %lf\n", a4, b41+b42+b43);
  //printf("a5 b5 %lf %lf\n", a5, b51+b52+b53+b54);
  //printf("a6 b6 %lf %lf\n", a6, b61+b62+b63+b64+b65);
  //printf("R5 %lf\n", R51+R53+R54+R55 );
  //printf("R6 %lf\n", R61+R63+R64+R65+R66 );

  if( a > b ) {
    printf("ERROR: RKF: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: RKF: a = b.\n");
  }


  if( hmin <= 0.0 ) {
    printf("ERROR: RKF: hmin <= 0.\n");
    return -1; 
  }

  if( hmin <= 1.0e-16 ) {
    printf("RARO: RKF: hmin <= 1.0e-16.\n");
  }

  if( h0 <= 0 ) {
    printf("ERROR: RKF: h0 <= 0.\n");
    return -1; 
  }

  if( h0 > b-a ) {
    printf("RARO: RKF: h0 > b-a.\n");
    h0 = b-a;
  }

  if( h0 < hmin ) {
    printf("RARO: RKF: h0 < hmin.\n");
    h0 = hmin;
  }

  if( nmax <= 0 ) {
    printf("ERROR: RKF: nmax <= 0.\n");
    return -1;
  }

  if( nmax == 1 ) {
    printf("RARO: RKF: nmax = 1.\n");
  }

  h = h0;
  n = 1;
  X[0] = a;
  Y[0] = ya;
  if( n == nmax ) {
    if( a == b ) res = 1;
    else res = 2;
    return res; 
  }
  
  x = a;
  y = ya;


  //printf("  h          y5          y6       e");
  //printf("         hbien  C    h_nuevo\n");

  while( x < b-eps0 ) {
    if( h > b-x ) h = b-x;
    if( n >= nmax ) return 2;
    hbien = 0;

    while( hbien == 0 ) {

      K1 = h*f(x,y);
      K2 = h*f(x+a2*h, y+b21*K1);
      K3 = h*f(x+a3*h, y+b31*K1+b32*K2);
      K4 = h*f(x+a4*h, y+b41*K1+b42*K2+b43*K3);    
      K5 = h*f(x+a5*h, y+b51*K1+b52*K2+b53*K3+b54*K4);    
      K6 = h*f(x+a6*h, y+b61*K1+b62*K2+b63*K3+b64*K4+b65*K5);    

      y5 = y + R51*K1 + R53*K3 + R54*K4 + R55*K5;
      y6 = y + R61*K1 + R63*K3 + R64*K4 + R65*K5 + R66*K6;


      e = fabs(y5-y6)/h;

      if( e <= eps ) {
        x = x+h;
        y = y6;
        X[n] = x;
        Y[n] = y;
        n++;
        hbien = 1;
      }

      C = 0.84*pow(eps/e, 0.25);
      if( C > 4.0 ) C = 4.0;
      if( C < 0.1 ) C = 0.1;
      hv = h; 
      h = C*hv;

      //printf("%10.6lf %10.6lf %10.6lf %8.4e ",hv,y5,y6,e);      
      //printf("%d %lf %lf\n",hbien,C,h);

      if( h < hmin ) {
        // h muy pequeno
        return 0;
      }   
    } // fin while hbien == 0

  } // fin while  x < b 

  return 1;
}
//----------------------------------------------------------
int AdamsB(double (*f)(double x, double y), double a, double b, 
           double ya, int m, int n, double *Y)
{
  // metodo de Adams-Bashforth de orden  m  
  // para la ecuacion diferencial
  // 
  //   y' = f(x,y)  en el intervalo  [a, b]
  //   y(a) = ya
  //
  // con  n >= 1  subintervalos y
  // 
  // con  0 <= m <= 4.
  
  // El orden del metodo, m, indica el grado del polinomio
  // de interpolacion usado.
  //
  // Devuelve   1  si todo el proceso se desarrolla bien.
  //            0  si la memoria es insuficiente
  //           -1  si hay errores en los datos.

  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  //  Y[i] sera la aproximacion de  y( a + i h ), i = 0,1,2,...,n
  //
  //  donde
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  n+1  valores.


  // coeficentes para m = 0,...,4
  double coef[] = {1.0, 
                   -1.0/2, 3.0/2, 
                   5.0/12,  -16.0/12, 23.0/12,
                   -9.0/24, 37.0/24, -59.0/24, 55.0/24,
                   251.0/720, -1274.0/720, 2616.0/720, -2774.0/720, 1901.0/720};

  // indica la cabeza o posicion del primer coeficiente para cada grado, 
  // p.e., para m = 4, el primer coeficente es coef[10].
  int cab[] = {0, 1, 3, 6, 10};
  double *c[5];
  double *F;

  double h, b1, s;
  int i, n1, res1, j, i1, im;

  if( a > b ) {
    printf("ERROR: AdamsB: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: AdamsB: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: AdamsB: n <= 0.\n");
    return -1; 
  }

  if( m < 0 || m > 4) {
    printf("ERROR: AdamsB: m  inadecuado.\n");
    return -1; 
  }

  // para almacenar los valores  f(xi,yi)
  F = new double[n+1];
  if( F == NULL ) return 0;

  h = (b-a)/double(n);

  for( i = 0; i <= 4; i++) c[i] = &coef[cab[i]];

  // una verificacion: suma = 1  
  /*
  for(  i = 0; i <= 4; i++) {
    s = 0.0;
    for(j = cab[i]; j <= cab[i]+i; j++) s += coef[j];
    printf("suma %lf\n", s);
  }
  */

  // otra vez la misma verificacion
  /*
  for(  i = 0; i <= 4; i++) {
    s = 0.0;
    for(j = 0; j <= i; j++) s += c[i][j];
    printf("suma %lf\n", s);
  }
  */

  n1 = m;
  if( n1 > n ) n1 = n;
  if( m > 0 ) {
    b1 = a + double(n1)*h;
    res1 = RK4(f, a, b1, ya, n1, Y);
    if( res1 == -1 ) printf("RARO: AdamsB: res1 = -1.\n");
  }
  else Y[0] = ya;
 
  for ( i = 0; i <= n1; i++ ) F[i] = f( a+double(i)*h, Y[i]);
  
  for( i = n1; i <= n-1; i++) {
    i1 = i+1;
    im = i-m;
    s = 0.0;
    for( j = 0; j <= m; j++ ) {
      s += c[m][j]*F[ im+j ];
    }
    Y[i1] = Y[i]+ s*h;
    F[i1] = f( a + double(i1)*h, Y[i1] );
  }
  return 1;
}
//----------------------------------------------------------
int AdamsM(double (*f)(double x, double y), double a, double b, 
           double ya, int m, int n, double *Y, double eps)
{
  // metodo de Adams-Moulton de orden  m
  // (predictor corrector)  
  //
  // para la ecuacion diferencial
  // 
  //   y' = f(x,y)  en el intervalo  [a, b]
  //   y(a) = ya
  //
  // con  n >= 1  subintervalos y
  // 
  // con  1 <= m <= 4.
  
  // El orden del metodo, m, indica el grado del polinomio
  // de interpolacion usado.
  //
  // Devuelve   1  si todo el proceso se desarrolla bien.
  //            0  si la memoria es insuficiente
  //           -1  si hay errores en los datos.
  //            2  si no hubo convergencia en el proceso
  //               predictor-corrector en el numero maximo
  //               de iteraciones previsto.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  //  Y[i] sera la aproximacion de  y( a + i h ), i = 0,1,2,...,n
  //
  //  donde
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  n+1  valores.


  // coeficentes para  m = 1,...,4
  double coef[] = {0.0, 
                   1.0/2, 1.0/2, 
                   -1.0/12, 8.0/12, 5.0/12,
                   1.0/24, -5.0/24, 19.0/24, 9.0/24,
                   -19.0/720, 106.0/720, -264.0/720, 646.0/720, 251.0/720};

  // indica la cabeza o posicion del primer coeficiente para cada grado, 
  // p.e., para m = 4, el primer coeficente es coef[10].
  int cab[] = {0, 1, 3, 6, 10};
  double *c[5];
  double *F;

  double h, b1, s, yi1k, xi1, yi1k1, ch;
  int i, n1, res1, j, i1, im, k;

  //*****************
  int maxit = 10;
  //*****************

  if( a > b ) {
    printf("ERROR: AdamsM: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: AdamsM: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: AdamsM: n <= 0.\n");
    return -1; 
  }

  if( m < 1 || m > 4) {
    printf("ERROR: AdamsM: m  inadecuado.\n");
    return -1; 
  }

  // para almacenar los valores  f(xi,yi)
  F = new double[n+1];
  if( F == NULL ) return 0;

  h = (b-a)/double(n);

  for( i = 0; i <= 4; i++) c[i] = &coef[cab[i]];

  // una verificacion: suma = 1  
  /*
  for(  i = 0; i <= 4; i++) {
    s = 0.0;
    for(j = cab[i]; j <= cab[i]+i; j++) s += coef[j];
    printf("suma %lf\n", s);
  }
  */

  // otra vez la misma verificacion
  /*
  for(  i = 0; i <= 4; i++) {
    s = 0.0;
    for(j = 0; j <= i; j++) s += c[i][j];
    printf("suma %lf\n", s);
  }
  */

  n1 = m-1;
  if( n1 > n ) n1 = n;

  if( m > 1 ) {
    b1 = a + double(n1)*h;
    res1 = RK4(f, a, b1, ya, n1, Y);
    if( res1 == -1 ) printf("RARO: AdamsM: res1 = -1.\n");
  }
  else Y[0] = ya;
 
  for ( i = 0; i <= n1; i++ ) F[i] = f( a+double(i)*h, Y[i]);
  
  for( i = n1; i <= n-1; i++) {
    i1 = i+1;
    xi1 = a + double(i1)*h;
    im = i1-m;
    yi1k = Y[i] + h*F[i];
    //printf("yi1k = %lf\n", yi1k);
    s = 0.0;
    for( j = 0; j < m; j++ ) {
      s += c[m][j]*F[ im+j ];
    }
    s = Y[i] + s*h;
    //printf("s = %lf\n", s);
    ch = h*c[m][m];
    for( k = 1; k <= maxit; k++ ) {
      yi1k1 = s + ch*f(xi1,yi1k);
      //printf("yi1k1 = %lf\n", yi1k1);
      if( fabs(yi1k1-yi1k) <= eps) {
        Y[i1] = yi1k1;
        break;
      } 
      yi1k = yi1k1;   
    }
    if( k > maxit ) return 2;

    F[i1] = f( a + double(i1)*h, Y[i1] );
  }
  return 1;
}
//----------------------------------------------------------
int RK4Sist(void (*f)(double x, double *y, double *Dy), int m, 
        double a, double b, double *ya, int n, double *Y)
{
  // metodo de Runge-Kutta de orden 4 para el sistema de
  // m  ecuaciones diferenciales 
  // 
  //   y' = f(x,y)  en el intervalo  [a, b]
  //   y(a) = ya
  //
  // con  n >= 1  subintervalos.

  // Como se trata de un sistema de ecuaciones diferenciales,
  // entonces  y, y', y(a) = ya  son vectores.

  // Devuelve   1  si los datos est� bien,
  //            0  si no hay mememoria suficiente
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  // primero estan los valores o las aproximaciones de 
  //                            y1(a)    y2(a)    ... ym(a)
  // despues                    y1(a+h)  y2(a+h)  ... ym(a+h)
  // ...
  //                            y1(a+ih) y2(a+ih) ... ym(a+ih)
  // ...
  // finalmente                 y1(b)    y2(b)        ym(b)               
  //                            
  //  donde        0 <= i <= n
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  (n+1)*m  valores.

  double h, *K1, *K2, *K3, *K4, xi, *yi, h2, *t;
  int i, im, j;

  if( a > b ) {
    printf("ERROR: RK4Sist: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: RK4Sist: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: RK4Sist: n <= 0.\n");
    return -1; 
  }

  if( m <= 0 ) {
    printf("ERROR: RK4Sist: m <= 0.\n");
    return -1; 
  }

  K1 = new double[m];
  K2 = new double[m];
  K3 = new double[m];
  K4 = new double[m];
  t  = new double[m];
  if(K1==NULL||K2==NULL||K3==NULL||K4==NULL||t==NULL)return 0;

  h = (b-a)/double(n);
  h2 = h/2.0;
  XIglY(Y, ya, m);
  for( i = 0; i < n; i++) {
    im = (i+1)*m;
    xi = a + double(i)*h;
    yi = &Y[i*m];

    f(xi, yi, K1);
    alfaX(h, K1, m);
    //printf("K1 : "); escrVect(K1, m);
    
    ZIglXMasAlfaY(t, yi, 0.5, K1, m);
    f( xi + h2, t, K2);
    alfaX(h, K2, m);
    //printf("K2 : "); escrVect(K2, m);

    ZIglXMasAlfaY(t, yi, 0.5, K2, m);
    f( xi + h2, t, K3);
    alfaX(h, K3, m);
    //printf("K3 : "); escrVect(K3, m);

    sumaXY(yi, K3, t, m);
    f( xi + h, t, K4);
    alfaX(h, K4, m);
    //printf("K4 : "); escrVect(K4, m);

    for(j=0;j<m;j++) Y[im+j] = yi[j]+(K1[j]+2.0*K2[j]+2.0*K3[j]+K4[j])/6.0; 

  }
  delete K1; delete K2; delete K3; delete K4; delete t;
  return 1;
}
//----------------------------------------------------------
int RK4m(double (*f)(double x, double *Diy), int m, 
         double a, double b, double *Diya, int n, double *Y)
{
  // metodo de Runge-Kutta de orden 4 para una ecuacion
  // diferencial de orden  m.
  //
  // Por ejemplo, si  m = 5, 
  // 
  //   y'''''   = f(x,y,y',y'',y''',y'''')  en el intervalo  [a, b]
  //   y(a)     = ya
  //   y'(a)    = y'a
  //   y''(a)   = y''a
  //   y'''(a)  = y'''a
  //   y''''(a) = y''''a
  //
  // con  n >= 1  subintervalos.

  // La ecuacion diferencial se puede escribir de manera
  // mas compacta
  //              
  //  D^m y = f(x, Di_y ) con   a <= x <= b
  // 
  //  Di_y(a) = Di_ya
  //
  //  donde Di_y  es un vector con  m  entradas
  //                 y, y', y'', ..., y^(m-1)
  //                 
  //  Di_ya  es el vector de condiciones iniciales.
  //       

  // Devuelve   1  si los datos estan bien,
  //            0  si no hay mememoria suficiente
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  // primero estan los valores o las aproximaciones de 
  //                            y(a)    y'(a)    ... y^(m-1)(a)
  //
  // despues                    y(a+h)  y'(a+h)  ... y^(m-1)(a+h)
  // ...
  //                            y(a+ih) y'(a+ih) ... y^(m-1)(a+ih)
  // ...
  //
  // finalmente                 y(b)    y'(b)        y^(m-1)(b)               
  //                            
  //  donde        0 <= i <= n
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  (n+1)*m  valores.

  double h, *K1, *K2, *K3, *K4, xi, *yi, h2;
  int i, im, j;

  if( a > b ) {
    printf("ERROR: RK4Sist: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: RK4Sist: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: RK4Sist: n <= 0.\n");
    return -1; 
  }

  if( m <= 0 ) {
    printf("ERROR: RK4Sist: m <= 0.\n");
    return -1; 
  }

  K1 = new double[m];
  K2 = new double[m];
  K3 = new double[m];
  K4 = new double[m];
  if(K1==NULL||K2==NULL||K3==NULL||K4==NULL)return 0;

  h = (b-a)/double(n);
  h2 = h/2.0;
  XIglY(Y, Diya, m);
  for( i = 0; i < n; i++) {
    im = (i+1)*m;
    xi = a + double(i)*h;
    yi = &Y[i*m];

    K1234m(f, h, xi, yi, 0.0, yi, K1, m);
    //printf("K1 : "); escrVect(K1, m);
    
    K1234m(f, h, xi+h2, yi, 0.5, K1, K2, m);
    //printf("K2 : "); escrVect(K2, m);

    K1234m(f, h, xi+h2, yi, 0.5, K2, K3, m);
    //printf("K3 : "); escrVect(K3, m);

    K1234m(f, h, xi+h, yi, 1.0, K3, K4, m);
    //printf("K4 : "); escrVect(K4, m);
 
    for(j=0;j<m;j++) Y[im+j] = yi[j]+(K1[j]+2.0*K2[j]+2.0*K3[j]+K4[j])/6.0; 

  }
  delete K1; delete K2; delete K3; delete K4;
  return 1;
}

//----------------------------------------------------------
int RK4_2(double (*f)(double x, double y, double Dy), double a, 
          double b, double ya, double Dya, int n, double *Y)
{
  // metodo de Runge-Kutta de orden 4 para una ecuacion
  // diferencial de orden  2. 
  // 
  //   y''   = f(x,y,y')  en el intervalo  [a, b]
  //   y(a)     = ya
  //   y'(a)    = Dya
  //
  // con  n >= 1  subintervalos.

  // Devuelve   1  si los datos est� bien,
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  // primero estan los valores o las aproximaciones de 
  //                            y(a)    y'(a)  
  //
  // despues                    y(a+h)  y'(a+h)
  // ...
  //                            y(a+ih) y'(a+ih)
  // ...
  //
  // finalmente                 y(b)    y'(b)                     
  //                            
  //  donde        0 <= i <= n
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  2*(n+1)  valores.

  double h, K1[2], K2[2], K3[2], K4[2], xi, yi[2], h2;
  int i, im, j;

  if( a > b ) {
    printf("ERROR: RK4Sist: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: RK4Sist: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: RK4Sist: n <= 0.\n");
    return -1; 
  }


  h = (b-a)/double(n);
  h2 = h/2.0;
  Y[0] = ya; Y[1] = Dya;

  for( i = 0; i < n; i++) {
    im = (i+1)*2;
    xi = a + double(i)*h;
    XIglY(yi, &Y[2*i], 2);

    K1234_2(f, h, xi, yi, 0.0, yi, K1);
    //printf("K1 : "); escrVect(K1, 2);

    K1234_2(f, h, xi+h2, yi, 0.5, K1, K2);
    //printf("K2 : "); escrVect(K2, 2);

    K1234_2(f, h, xi+h2, yi, 0.5, K2, K3);
    //printf("K3 : "); escrVect(K3, 2);

    K1234_2(f, h, xi+h, yi, 1.0, K3, K4);
    //printf("K4 : "); escrVect(K4, 2);

    for(j=0;j<2;j++) Y[im+j] = yi[j]+(K1[j]+2.0*K2[j]+2.0*K3[j]+K4[j])/6.0; 
  }

  return 1;
}
//----------------------------------------------------------
int disparo(double (*f)(double x, double y, double Dy), double a, 
          double b, double ya, double yb, int n, double *Y, double eps)
{
  // metodo del disparo para ecuacion diferencial de segundo orden  
  // con condiciones de frontera
  // 
  //   y''   = f(x,y,y')  en el intervalo  [a, b]
  //   y(a)     = ya
  //   y(b)     = yb
  //
  // con  n >= 1  subintervalos.

  // Devuelve   1  si los datos estan bien,
  //            2  si en   maxit  iteraciones no se obtuvo convergencia
  //               con el  eps  pasado como parametro
  //            0  si el metodo no avanza sin llegar a la convergencia
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  // 
  // primero estan los valores o las aproximaciones de 
  //                            y(a)    y'(a)  
  //
  // despues                    y(a+h)  y'(a+h)
  // ...
  //                            y(a+ih) y'(a+ih)
  // ...
  //
  // finalmente                 y(b)    y'(b)                     
  //                            
  //  donde        0 <= i <= n
  //
  //               h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  2*(n+1)  valores.

  int k;
  double v0, v1, v2, fv0, fv1, yyb, delta;

  //*********************
  int maxit = 10;
  double eps0 = 1.0e-12;
  //*********************

  if( a > b ) {
    printf("ERROR: disparo: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: disparo: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: disparo: n <= 0.\n");
    return -1; 
  }

  v0 = (yb-ya)/(b-a);
  //printf("v0 = %lf\n", v0);

  RK4_2(f, a, b, ya, v0, n, Y);

  yyb = Y[2*n];
  //printf("aprox y(b) = %lf\n", yyb);

  fv0 = yb - yyb;
  //printf("fv0 = %e\n", fv0);

  if( fabs(fv0) <= eps ) return 1;

  v1 = v0 + fv0/(b-a);
  //printf("v1 = %lf\n", v1);

  for( k = 0; k <= maxit; k++) {
    RK4_2(f, a, b, ya, v1, n, Y);
    yyb = Y[2*n];
    //printf("aprox y(b) = %lf\n", yyb);
    fv1 = yb - yyb;
    //printf("fv1 = %e\n", fv1);
    if( fabs(fv0) <= eps ) return 1;

    delta = fv1-fv0;
    if( fabs(delta) <= eps0 ) return 0;
    
    v2 = v1 - fv1*(v1-v0)/delta;
    v0 = v1;
    v1 = v2;
    fv0 = fv1; 
  }

  return 1;
}
//----------------------------------------------------------
int DF_CF(void (*f)(double x, double *pqrs), double a, 
          double b, double ya, double yb, int n, double *Y)
{
  // metodo de diferencias finitas para ecuacion diferencial
  // lineal de segundo orden con condiciones de frontera
  // 
  //   p(x) y'' + q(x) y' + r(x) y = s(x)    a <= x <= b
  //   y(a)     = ya
  //   y(b)     = yb
  //
  // con  n >= 1  subintervalos.

  // Devuelve   1  si todo funciono bien
  //            2  si el sistema tridiagonal no se pudo resolver
  //            0  si no hay suficiente memoria
  //           -1  si hay errores en los datos.
  //
  // Los valores calculados estaran en el arreglo  Y
  // de la siguiente forma:
  //
  //  Y[i] sera la aproximacion de  y( a + i h ), i = 0,1,2,...,n 
  //
  //  con             h = (b-a)/n             
  //
  // El arreglo o apuntador  Y  debe estar previsto para almacenar
  // por lo menos  n+1  valores.

  double *d, *u, *L, *bt;
  double pqrs[4];
  double h, h22, pi, qi, ri, si;
  int i, n1, i1, res, n2;

  //******************
  double eps = 1.0e-8;
  //******************

  if( a > b ) {
    printf("ERROR: disparo: a > b.\n");
    return -1; 
  }

  if( a == b ) {
    printf("RARO: disparo: a = b.\n");
  }

  if( n <= 0 ) {
    printf("ERROR: disparo: n <= 0.\n");
    return -1; 
  }

  Y[0] = ya;
  Y[n] = yb;

  n1 = n-1;  
  n2 = n-2;

  if( n1 == 0 ) {
    printf("RARO: DF_CF: n = 1.\n");
    return 1;
  }

  d = new double[n1];
  if(d==NULL) return 0;
  if( n2 > 0 ) {
    u = new double[n2];
    L = new double[n2];
    if(u==NULL||L==NULL) return 0;
  }

  

  bt= &Y[1];

  h = (b-a)/double(n);
  h22= 2*h*h;

  if( n1 > 1 ) {
    for( i = 1; i <= n1; i++ ) {
      i1 = i-1;
      f( a + double(i)*h, pqrs);
      pi = pqrs[0];
      qi = pqrs[1];
      ri = pqrs[2];
      si = pqrs[3];
    
      if( 2 <= i && i <= n-2 ) {
        d[i1] = -4*pi + h22*ri;
        u[i1]  = 2*pi + h*qi;
        L[i-2] = 2*pi - h*qi;
        bt[i1] = h22*si;
      }
      else if( i == 1 ) {
        d[i1] = -4*pi + h22*ri;
        u[i1] =  2*pi + h*qi;
        bt[i1] = h22*si - (2*pi-h*qi)*ya;
      }
      else {
        // i = n-1
        d[i1] = -4*pi + h22*ri;
        bt[i1] = h22*si - (2*pi + h*qi)*yb;
        L[i-2] = 2*pi - h*qi;
      }    
    } // fin for i

    // fin n1 > 1
  }
  else {
    // n1 = 1 
    f( a+h, pqrs);
    pi = pqrs[0];
    qi = pqrs[1];
    ri = pqrs[2];
    si = pqrs[3];
    d[0] = ri*h22 - 4*pi;
    bt[0] = si*h22 -h*qi*(yb-ya) -2*pi*(yb+ya);
  }

  //printf("d :  ");escrVect(d,n1);printf("L :  ");escrVect(L,n-2);
  //printf("u :  ");escrVect(u, n-2);printf("beta:");escrVect(bt,n1);

  res = tridiag(d, L, u, bt, n1, eps);  
  if( res == 0 ) return 2;  
  else return 1;

}
//----------------------------------------------------------
