//  SOLUCION DE ECUACIONES

//  ecuac.h

// NECESITA 

#include <math.h>

double Newton(double (*fxDfx)(double x, double &Dfx), double x0, 
              int maxit, int &indic, int &k, double epsF, 
              double epsX, double eps0);

double secante(double (*f)(double x), double x0, int maxit, 
               int &indic, int &k, double epsF, double epsX, 
               double eps0);

double bisec(double (*f)(double x), double a, double b, 
             int &indic, int &k, double epsX, double epsF);

double RegulaFMod(double (*f)(double x), double a, double b, 
                  int &indic, int &k, double epsX, double epsF);

double puntoFijo(double (*g)(double x), double x0, int maxit, 
                 int &indic, int &k, double epsX);

double modPuntoF(double (*g)(double x), double x0, int maxit, 
                 int &indic, int &k, double epsX);

//----------------------------------------------------------
double Newton(double (*fxDfx)(double x, double &Dfx), double x0, 
              int maxit, int &indic, int &k, double epsF, 
              double epsX, double eps0)
{
  // Metodo de Newton para resolver  f(x) = 0.
  //
  // f(x)  es el valor devuelto por la funcion  fxDfx.
  // Esta ultima funcion tambien calcula f'(x).

  //
  // indic  valdra:
  //    1  si  | f(xk) | <= epsF
  //    2  si  | xk-xk1 |/( 1+ |xk| ) <= epsX
  //    0  si  | f'(xk ) | <= eps0
  //    3  si  en  maxit  iteraciones no se ha tenido
  //       convergencia
  //
  // Sea  r  el valor devuelto por  Newton.
  // Si indic = 1,  r  es una buena aproximacion de una raiz
  // Si indic = 2,  r  es una buena aproximacion de una raiz
  //                   o el ultimo  xk  calculado.
  //          = 3,  r  es el ultimo  xk  calculado.
  //          = 0,  r  es el ultimo  xk  calculado.

  // x0  es una aproximacion inicial de la raiz.

  // k  indicara el numero de iteraciones realizadas.

  double delta, xk, fx, Dfx;

  xk = x0;
  fx = fxDfx(xk, Dfx);
  //printf("%2d %20.8lf %13e %20.8lf\n", 0, xk, fx, Dfx);

  for( k=1; k<= maxit; k++) {
    if( fabs(Dfx) <= eps0 ) {
      indic = 0;
      return xk;
    }
    delta = fx/Dfx;
    xk = xk - delta;
    fx = fxDfx(xk, Dfx);
    if( fabs(fx) <= epsF ) {
      indic = 1;
      return xk;
    }
    //printf("%2d %20.8lf %13e %20.8lf\n", k, xk, fx, Dfx);
    if( fabs(delta)/( 1.0+fabs(xk) ) <= epsX ) {
      indic = 2;
      return xk;
    }
  }
  indic = 3;
  return xk;
}
//----------------------------------------------------------
double secante(double (*f)(double x), double x0, int maxit, 
               int &indic, int &k, double epsF, double epsX, 
               double eps0)
{
  // Metodo de la  secante  para resolver  f(x) = 0.
  //
  // f(x)  esta definida en la funcion  f
  //
  // indic  valdra:
  //    1  si  | f(x2) | <= epsF
  //    2  si  | x2-x1 |/( 1+ |x2| ) <= epsX
  //    0  si  | f(x1) - f(x0) | <= eps0
  //    3  si  en  maxit  iteraciones no se ha tenido
  //       convergencia
  //
  // Sea  r  el valor devuelto por  secante.
  // Si indic = 1,  r  es una buena aproximacion de una raiz
  // Si indic = 2,  r  es una buena aproximacion de una raiz
  //                   o el ultimo  x2  calculado.
  //          = 3,  r  es el ultimo  x2  calculado.
  //          = 0,  r  es el ultimo  x2  calculado.
  // x0  es una aproximacion inicial de la raiz.

  // k  indicara el numero de iteraciones realizadas.

  double delta, x1, x2, f0, f1, f2, den, h;

  //********
  h = 0.01;
  //********

  x1 = x0 + h;

  f0 = f(x0);
  f1 = f(x1);
  //printf(" 0 %10.6lf %13e\n", x0, f0);
  //printf(" 1 %10.6lf %13e\n", x1, f1);

  for( k=1; k<= maxit; k++) {
    den = f1 - f0;
    if( fabs(den) <= eps0 ) {
      indic = 0;
      return x1;
    }
    delta = f1*(x1-x0)/den;
    x2 = x1 - delta;
    f2 = f(x2);
    //printf("%2d %10.6lf %13e\n", k+1, x2, f2);
    if( fabs(f2) <= epsF ) {
      indic = 1;
      return x2;
    }
    if( fabs(delta)/( 1.0+fabs(x2) ) <= epsX ) {
      indic = 2;
      return x2;
    }
    x0 = x1;  x1 = x2;  f0 = f1;  f1 = f2;
  }
  indic = 3;
  return x2;
}
//----------------------------------------------------------
double bisec(double (*f)(double x), double a, double b, 
             int &indic, int &k, double epsX, double epsF)
{
  // Metodo de biseccion para  f(x) = 0.
  // f debe ser continua en [a,b] y f(a)f(b) < 0.

  // Esta funcion calcula dos valores:  r ,  indic
  // Devuelve  r,  posiblemente una raiz.
  // indic   es un parametro por referencia.

  // indic  valdra:
  //     1   si se obtuvo una raiz,
  //     0   si f(a)f(b) > 0.0
  //    -1   si a >= b
  //    -2   si epsX  <= 0  o  epsF < 0.0

  // Si  indic = 1, el valor devuelto sera una raiz.
  // En los demas casos el valor devuelto sera  0.0
  // Sea  x  una raiz exacta;  r  es raiz (aproximada)
  // si |f(r)| <= epsF  o   si  |r-x| <= epsX

  double fa, fb, m, fm;

  k = 0;

  if( epsX <= 0.0 || epsF < 0.0 ) {
    indic = -2;
    return 0.0;
  }
  fa = f(a);
  fb = f(b);
  if( fabs(fa) <= epsF ) {
    indic = 1;
    return a;
  }
  if( fabs(fb) <= epsF ) {
    indic = 1;
    return b;
  }
  if( a >= b ) {
    indic = -1;
    return 0.0;
  }
  if( fa*fb > 0.0 ) {
    indic = 0;
    return 0.0;
  }

  k = 0;
  while( b-a > epsX) {
    m = (a+b)/2.0;
    fm = f(m);
    //printf("%10.6lf %10.6lf\n",m,fm);
    if( fabs(fm) <= epsF ) {
      indic = 1;
      return m;
    }
    if( fa*fm < 0.0 ) {
      // nuevo intervalo  [a,m]
      b = m;
      fb = fm;
    }
    else{
      // nuevo intervalo  [m,b]
      a = m;
      fa = fm;
    }
    k++;
    //printf("%2d%10.6lf %10.6lf%10.6lf%10.6lf",k,a,b,fa,fb);
  }
  indic = 1;
  if( fabs(fa) <= fabs(fb) ) return a;
  else return b;
}
//----------------------------------------------------------
double RegulaFMod(double (*f)(double x), double a, double b, 
                  int &indic, int &k, double epsX, double epsF)
{
  // Metodo de Regula Falsi MODIFICADO para  f(x) = 0.
  // f debe ser continua en [a,b] y f(a)f(b) < 0.

  // Esta funcion calcula dos valores:  r ,  indic
  // Devuelve  r,  posiblemente una raiz.
  // indic   es un parametro por referencia.

  // indic  valdra:
  //     1   si se obtuvo una raiz,
  //     0   si f(a)f(b) > 0.0
  //    -1   si a >= b
  //    -2   si epsX  <= 0  o  epsF < 0.0

  // Si  indic = 1, el valor devuelto sera una raiz.
  // En los demas casos el valor devuelto sera  0.0
  // Sea  x  una raiz exacta;  r  es raiz (aproximada)
  // si |f(r)| <= epsF  o   si  |r-x| <= epsX

  double fa, fb, m, fm, c, fc, epsX2, t, t1, t2, t3, gra;

  epsX2 = epsX/2.0;
  if( epsX <= 0.0 || epsF < 0.0 ) {
    indic = -2;
    return 0.0;
  }
  fa = f(a);
  fb = f(b);
  if( fabs(fa) <= epsF ) {
    indic = 1;
    return a;
  }
  if( fabs(fb) <= epsF ) {
    indic = 1;
    return b;
  }
  if( a >= b ) {
    indic = -1;
    return 0.0;
  }
  if( fa*fb > 0.0 ) {
    indic = 0;
    return 0.0;
  }

  gra = 1.0/epsX; // valor grande

  if( gra < b-a ) gra = b-a;

  k = 0;
  //printf(" k   a     b    f(a)   f(b)    c    f(c)    m    f(m)");
  while( b-a > epsX) {
    //printf("\n%2d %7.4lf %7.4lf %8.4lf %8.4lf",k,a,b,fa,fb);
    c = a -fa*(b-a)/(fb-fa);
    fc = f(c);
    //printf("   %7.4lf %8.4lf",c,fc);
    if( fabs(fc) <= epsF ) {
      indic = 1;
      return c;
    }
    m = (a+b)/2.0;
    fm = f(m);
    //printf("   %7.4lf %8.4lf",m,fm);
    if( fabs(fm) <= epsF ) {
      indic = 1;
      return m;
    }
    if( fabs(m-c) <= epsX2 ) {
      // biseccion sencilla
      if( fa*fm < 0.0 ) {
        // nuevo intervalo  [a,m]
        b = m;
        fb = fm;
      }
      else{
        // nuevo intervalo  [m,b]
        a = m;
        fa = fm;
      }
    }
    else{
      // 3 subintervalos
      // cambios para que  c < m
      if( c > m ) {
        t = c;
        c = m;
        m = t;
        t  = fc;
        fc = fm;
        fm = t;
      }
      if( fa*fc < 0.0 ) t1 = c-a;
      else t1 = gra;
      if( fc*fm < 0.0 ) t2 = m-c;
      else t2 = gra;
      if( fm*fb < 0.0 ) t3 = b-m;
      else t3 = gra;
      if( t1 <= t2 && t1 <= t3 ) {
        b  = c;
        fb = fc;
      }
      else{
        if( t2 <= t1 && t2 <= t3 ) {
          a  = c;
          fa = fc;
          b  = m;
          fb = fm;
        }
        else{
          a  = m;
          fa = fm;
        }
      }
    }
    k++;
    //printf("%2d%7.4lf%7.4lf%10.2e%10.2e",k,a,b,fa,fb);
  }
  indic = 1;
  if( fabs(fa) <= fabs(fb) ) return a;
  else return b;
}
//----------------------------------------------------------
double puntoFijo(double (*g)(double x), double x0, int maxit, 
                 int &indic, int &k, double epsX)
{
  // Metodo de punto fijo par resolver  g(x) = x
  // La funcion g  esta definida la funcion  g.

  // Sea  r  el valor devuelto.
  // Si indic = 1 ,  r es una aproximacion de la solucion
  // Si indic = 0 ,  r  es el ultimo valor obtenido;
  //                 hubo  demasiados aumentos consecutivos 
  //                 en  | x(k+1) - xk |
  // Si indic = 2 ,  r  es el ultimo valor obtenido;
  //                 hubo  demasiadas iteraciones.

  double dif0, dif1, x1;
  int nAum = 0;

  //***************
  int nMaxAum = 3; // numero maximo de aumentos consecutivos
  //***************

  dif0 = 1.0/epsX;
  indic = 1;
  for( k = 1; k <= maxit; k++) {
    x1 = g(x0);
    //printf("%20.6lf\n", x1);
    dif1 = fabs(x1-x0);

    if( dif1 <= epsX ) return x1;

    if( dif1 < dif0 ) nAum = 0;
    else nAum++;

    if( nAum >= nMaxAum ) {
      indic = 0;
      return x1;
    }
    dif0 = dif1;
    x0 = x1;
  }
  indic = 2;
  return x1;
}
//----------------------------------------------------------
double modPuntoF(double (*g)(double x), double x0, int maxit, 
                 int &indic, int &k, double epsX)
{
  // Modificacion de metodo de punto fijo par resolver  g(x) = x

  // x3 esta determinado p0r el punto de corte de
  // la recta  y = x
  // la recta que pasa por  (x0, x1), (x1, x2)

  // La funcion g  esta definida en la funcion  g.

  // Sea  r  el valor devuelto.
  // Si indic = 1 ,  r es una aproximacion de la solucion
  // Si indic = 0 ,  r  es el ultimo valor obtenido;
  //                 hubo  demasiados aumentos consecutivos 
  //                 en  | x(k+1) - xk |
  // Si indic = 2 ,  r  es el ultimo valor obtenido;
  //                 hubo  demasiadas iteraciones.

  double dif0, dif1, x1, x2, x10, x21, x3, m;
  int nAum = 0;

  //***************
  int nMaxAum = 3; // numero maximo de aumentos consecutivos
  //***************

  dif0 = 1.0/epsX;
  indic = 1;
  for( k = 1; k <= maxit; k++) {
    x1 = g(x0);
    //printf("%20.6lf\n", x1);
    x10 = x1-x0;
    if( fabs(x10) <= epsX ) return x1;

    x2 = g(x1);
    x21 = x2-x1;
    if( fabs(x21) <= epsX ) return x2;

    m = x21/x10;
    x3 = (m*x0-x1)/(m-1.0);
    dif1 = fabs(x3-x1);

    if( dif1 < dif0 ) nAum = 0;
    else nAum++;

    if( nAum >= nMaxAum ) {
      indic = 0;
      return x3;
    }
    dif0 = dif1;
    x0 = x3;
  }
  indic = 2;
  return x3;
}
//----------------------------------------------------------
double f1(double x, double &Dfx)
{
  // devuelve f(x)  y  calcula Dfx = f'(x)

  Dfx = 5.0*x*x*x*x-12.0*x*x*x+10.0; 
  return x*x*x*x*x-3.0*x*x*x*x+10.0*x-8.0;
}
//----------------------------------------------------------
double f2(double x)
{
  // devuelve f(x)

  return x*x*x*x*x-3.0*x*x*x*x+10.0*x-8.0;
}
//----------------------------------------------------------
double g3(double x)
{
  // devuelve g(x)

  return (x + 3.0/x)/2.0;
}
//----------------------------------------------------------
double g4(double x)
{
  // devuelve g(x)

  return -(x*x*x*x*x-3.0*x*x*x*x-8.0)/10.0;
}
